/* ************************************************************************
 *
 *   global configuration, setup and settings
 *
 *   (c) 2012-2016 by Markus Reschke
 *   based on code from Markus Frejek and Karl-Heinz Kbbeler
 *
 * ************************************************************************ */


/* source management */
#define CONFIG_H


/*
 *  For MCU specific settings (port and pin assignments) and LCD display
 *  settings please edit:
 *  - ATmega328:           config_328.h
 *  - ATmega324/644/1284:  config_644.h (not supported yet)
 */



/* ************************************************************************
 *   Hardware options
 * ************************************************************************ */


/*
 *  rotary encoder for user interface (default: PD2 & PD3)
 *  - in parallel with LCD module
 *  - see ENCODER_PORT for port pins
 *  - uncomment to enable and also set ENCODER_PULSES below to match your
 *    rotary encoder
 */

//#define HW_ENCODER


/*
 *  Number of pulses per step or detent for the rotary encoder
 *  - typical values: 1, 2 or 4
 *  - adjust value to match your rotary encoder
 */

#define ENCODER_PULSES   2


/*
 *  2.5V voltage reference for Vcc check (default: PC4)
 *  - should be at least 10 times more precise than the voltage regulator
 *  - see TP_REF for port pin
 *  - uncomment to enable and also adjust UREF_25 below for your voltage
 *    reference
 */

//#define HW_REF25


/*
 *  Typical voltage of 2.5V voltage reference (in mV)
 *  - see datasheet of the voltage reference
 *  - or use >= 5.5 digit DMM to measure the voltage
 */

#define UREF_25           2495


/*
 *  Probe protection relay for discharging caps (PC4):
 *  - low signal: short circuit probe pins
 *    high signal via external reference: remove short circuit 
 *  - uncomment to enable
 */

//#define HW_RELAY


/*
 *  voltage measurement up to 50V DC (default: PC3):
 *  - 10:1 voltage divider
 *  - for Zener diodes
 *  - DC-DC boost converter controled by test push button
 *  - see TP_BAT for port pin
 *  - uncomment to enable
 */

//#define HW_ZENER


/*
 *  frequency counter (default: PD4)
 *  - in parallel with LCD module
 *  - requires MCU with >=32kB Flash
 *  - uncomment to enable
 */

//#define HW_FREQ_COUNTER



/* ************************************************************************
 *   software options
 * ************************************************************************ */


/*
 *  PWM generator
 *  - uncomment to enable
 */

#define SW_PWM


/*
 *  Inductance measurement
 *  - uncomment to enable
 */

#define SW_INDUCTOR


/*
 *  ESR measurement and in-circuit ESR measurement
 *  - uncomment to enable
 */

#define SW_ESR


/*
 *  Check for rotary encoders
 *  - uncomment to enable
 */

#define SW_ENCODER


/*
 *  squarewave signal generator
 *  - requires rotary encoder
 *  - uncomment to enable
 */

#define SW_SQUAREWAVE


/*
 *  IR remote control detection/decoder
 *  - requires IR receiver module, e.g. TSOP series
 *  - uncomment to enable
 */

#define SW_IR_RECEIVER


/*
 *  current limiting resistor for IR receiver module
 *  - for 5V only modules
 *  - Warning: any short circuit may destroy your MCU
 *  - uncomment to disable resistor
 */

//#define SW_IR_DISABLE_RESISTOR


/*
 *  check for opto couplers
 *  - uncomment to enable
 */

#define SW_OPTO_COUPLER


/*
 *  check for Unijunction Transistors
 *  - uncomment to enable
 */

#define SW_UJT



/* ************************************************************************
 *   Makefile workaround for some IDEs 
 * ************************************************************************ */


/*
 *  Oscillator startup cycles (after wakeup from power-safe mode):
 *  - typical values
 *    - internal RC:              6
 *    - full swing crystal:   16384 (also 256 or 1024 based on fuse settings)
 *    - low power crystal:    16384 (also 256 or 1024 based on fuse settings)
 *  - Please change value if it doesn't match your tester!
 */

#ifndef OSC_STARTUP
  #define OSC_STARTUP    16384
#endif



/* ************************************************************************
 *   misc settings
 * ************************************************************************ */


/*
 *  Languange of user interface. Available languages:
 *  - English (default)
 *  - German
 *  - Czech
 */

#define UI_ENGLISH
//#define UI_GERMAN
//#define UI_CZECH


/*
 *  LCD module with cyrillic character set
 *  - uncomment if you are using such an LCD
 *  - obsolete
 */

//#define LCD_CYRILLIC


/*
 *  Maximum time to wait after a measurement in continous mode (in ms).
 *  - Time between printing the result and starting a new cycle.
 */

#define CYCLE_DELAY      3000


/*
 *  Maximum number of measurements without any components found.
 *  - If that number is reached the tester powers off.
 */

#define CYCLE_MAX        5


/*
 *  Voltage divider for battery monitoring
 *  - BAT_R1: top resistor in Ohms
 *  - BAT_R2: bottom resistor in Ohms
 *
 */

#define BAT_R1          10000
#define BAT_R2          3300


/*
 *  Voltage drop by reverse voltage protection diode and power management
 *  transistor (in mV):
 *  - Schottky diode about 200mV / Transistor about 100mV.
 *  - Get your DMM and measure the voltage drop!
 *  - Could be also used to compensate any offset by the voltage divider
 *    used to measure the battery voltage.
 */  

#define BAT_OFFSET       290


/*
 *  Battery low voltage (in mV).
 *  - Tester warns if BAT_POOR + 1V is reached.
 *  - Tester powers off if BAT_POOR is reached.
 *  - Voltage drop (BAT_OUT) is considered in calculation.
 */

#define BAT_POOR         6400



/* ************************************************************************
 *   measurement settings and offsets
 * ************************************************************************ */


/*
 *  ADC voltage reference based on Vcc (in mV). 
 */

#define UREF_VCC         5001


/*
 * Offset for the internal bandgap voltage reference (in mV): -100 up to 100
 *  - To compensate any difference between real value and measured value.
 *  - The ADC has a resolution of about 4.88mV for V_ref = 5V (Vcc) and
 *    1.07mV for V_ref = 1.1V (bandgap).
 *  - Will be added to measured voltage of bandgap reference.
 */

#define UREF_OFFSET      0


/*
 *  Exact values of probe resistors.
 *  - Standard value for Rl is 680 Ohms.
 *  - Standard value for Rh is 470k Ohms.
 */

/* Rl in Ohms */
#define R_LOW            680

/* Rh in Ohms */
#define R_HIGH           470000


/*
 *  Offset for systematic error of resistor measurement with Rh (470k) 
 *  in Ohms.
 */

#define RH_OFFSET        700 



/*
 *  Resistance of probe leads (in 0.01 Ohms).
 *  - Resistance of two probe leads in series.
 *  - Assuming all probe leads got same/similar resistance.
 */

#define R_ZERO           20



/* 
 *  Capacitance of the wires between PCB and terminals (in pF).
 *  Examples:
 *  - 2pF for wires 10cm long
 */

#define CAP_WIRES        2


/* 
 *  Capacitance of the probe leads connected to the tester (in pF).
 *  Examples:
 *    capacity  length of probe leads
 *    -------------------------------
 *     3pF      about 10cm
 *     9pF      about 30cm
 *    15pF      about 50cm
 */

#define CAP_PROBELEADS   9


/*
 *  Maximum voltage at which we consider a capacitor being
 *  discharged (in mV)
 */

#define CAP_DISCHARGED   2


/*
 *  Number of ADC samples to perform for each mesurement.
 *  - Valid values are in the range of 1 - 255.
 */

#define ADC_SAMPLES      25



/* ************************************************************************
 *   MCU specific setup to support different AVRs
 * ************************************************************************ */


/*
 *  ATmega328
 */

#if defined(__AVR_ATmega328__)

  #include "config_328.h"


/*
 *  ATmega324 / ATmega664 / ATmega1284
 */

#elif defined(__AVR_ATmega324__) || defined(__AVR_ATmega644__) || defined(__AVR_ATmega1284__)

  #include "config_644.h"


/*
 *  missing or unsupported MCU
 */

#else
  #error <<< No or wrong MCU type selected! >>>
#endif



/* ************************************************************************
 *   ADC clock
 * ************************************************************************ */


/*
 *  ADC clock 
 *  - The ADC clock is 125000Hz by default.
 *  - You could also set 250000Hz, but that exceeds the max. ADC clock
 *    of 200kHz for 10 bit resolution!
 *  - Special case for MCU clock 20MHz: 156250Hz
 */

#define ADC_FREQ    125000


/*
 *  define clock divider
 *  - supports 1MHz, 2MHz, 4MHz, 8MHz and 16MHz MCU clocks
 *  - we got only 7 fixed prescalers from 2 up to 128
 */

#define CPU_FREQ    F_CPU

/* 1MHz/250kHz */
#if CPU_FREQ / ADC_FREQ == 4
  #define ADC_CLOCK_DIV (1 << ADPS1) 
#endif

/* 1MHz/125kHz 2MHz/250kHz */
#if CPU_FREQ / ADC_FREQ == 8
  #define ADC_CLOCK_DIV (1 << ADPS1) | (1 << ADPS0)
#endif

/* 2MHz/125kHz 4MHz/250kHz */
#if CPU_FREQ / ADC_FREQ == 16
  #define ADC_CLOCK_DIV (1 << ADPS2)
#endif

/* 4MHz/125kHz 8MHz/250kHz */
#if CPU_FREQ / ADC_FREQ == 32
  #define ADC_CLOCK_DIV (1 << ADPS2) | (1 << ADPS0)
#endif

/* 8MHz/125kHz 16MHz/250kHz */
#if CPU_FREQ / ADC_FREQ == 64
  #define ADC_CLOCK_DIV (1 << ADPS2) | (1 << ADPS1)
#endif

/* 16MHz/125kHz 20MHz/156.25kHz */
#if CPU_FREQ / ADC_FREQ == 128
  #define ADC_CLOCK_DIV (1 << ADPS2) | (1 << ADPS1) | (1 << ADPS0)
#endif



/* ************************************************************************
 *   derived values
 * ************************************************************************ */


/*
 *  total default capacitance (in pF)
 *  - max. 255
 */

#define C_ZERO           CAP_PCB + CAP_WIRES + CAP_PROBELEADS


/*
 *  number of MCU cycles per s
 *  - min. 1 (for 1MHz)
 *  - max. 20 (for 20MHz)
 */

#define MCU_CYCLES_PER_US     (CPU_FREQ / 1000000)


/*
 *  number of MCU cycles per ADC cycle
 *  - min. 4
 *  - max. 128
 */ 

#define MCU_CYCLES_PER_ADC    (CPU_FREQ / ADC_FREQ)



/* ************************************************************************
 *   ressource management
 * ************************************************************************ */


/*
 *  software options
 */

/* squarewave generator requires rotary encoder */
#ifdef SW_SQUAREWAVE
  #ifndef HW_ENCODER
    #undef SW_SQUAREWAVE
  #endif
#endif

/* LCD module */
#ifdef LCD_CONTRAST
  #define SW_CONTRAST
#else
  #define LCD_CONTRAST        0
#endif

/* component symbols for fancy pinout */
#if defined (SYMBOLS_24X24_V) || defined (SYMBOLS_24X24_H) || defined (SYMBOLS_32X32_H)
  #define SW_SYMBOLS
#endif

/* touchscreen */
#ifdef TOUCH_PORT
  #define HW_TOUCH
#endif



/* ************************************************************************
 *   EOF
 * ************************************************************************ */
