# SSD1306 SPI OLED

## A fast SSD1306 Arduino library for OLED displays (SPI only)

Having received my cheap 0.96" Serial 128X64 OLED display from China I tried to get it to display something.

According to the seller, the OLED display uses the SSD1306 driver IC. So after a quick search, I found that the Adafruit and u8glib libraries claimed to support the SSD1306.  However my hopes were short lived, as no matter what I tried, I was not able to get anything meaningful/useful displayed.

So it was back to basics and the SSD1306 datasheet (the application note at the end of the datasheet was very useful).  A few(?) lines of code later and text was finally appearing on the display.

Having used [Arthur Liberman's PCD8544_SPI library][1] with my Nokia 5110 LCD display, I thought I would use it as the basis for this SSD1306_SPI library.

[1]: http://forum.arduino.cc/index.php?PHPSESSID=5qoe2ghj5oloh918gta07ulph7&topic=176794.0

The library uses Arduino's SPI bus. It assumes your Arduino is running at 16MHz, and uses a /4 SPI (default) divider for an effective 4MHz SPI speed.

## Wiring

The library is configured to run on the Atmega168/328 (Uno, Mini, Nano, Duemilanove), but it can be reconfigured to run on any AVR based Arduino board.
By default it uses Digital Pins 8 - 13 (PORT B).

| OLED pin | Arduino pin
|------|------
| GND | GND
| VCC | 3.3v (or VCC)
| D0 | D13 (SCLK)
| D1 | D11 (MOSI)
| RES | D8
| DC | D9
| CS | D10

The SSD1306 is rated at 3.3v, so I connected the OLED display to an Arduino Pro Mini running at 3.3v.

**Note:** There is some mention on forums that people have had problems trying to run SSD1306 based OLED displays at 5v levels. You may need to use level shifters (e.g. CD4050) to get nice and clean 3.3v signals to the OLED.

## Code

There are two versions of this library.

One utilizes a frame buffer which uses 1024 bytes of SRAM on your device, but it gives you additional flexibility. It allows you to set an individual pixel on or off, it lets you draw a rectangle (empty or filled, empty is the default) and horizontal or vertical lines (an attempt to draw a line at an angle will be ignored).

The second version does not use a frame buffer, therefore the memory footprint of this library is not very large. The lack of a frame buffer may (or may not) give a small penalty in performance (depending on what you display) and only allows printing text and or bitmap data onto the screen. So if you don't need the setPixel, writeRect and writeLine functions, save your RAM and use the non-frame buffered version.

The feature set may be quite basic (compared to u8glib or Adafruit), but it's fast and more than what I currently need for my plans for this display.

It does not contain any protection against going out of the frame buffer (or the physical screen) so make sure you handle the screen borders properly.

### Usage

Call `begin()` in your `setup()` method.

Call the `print()` function like you do with the LiquidCrystal or Serial library. It only supports ASCII characters 0x20 - 0x7F (32 - 127). `print()` returns the number of bytes printed.

The `gotoXY` is used for printing text; it looks at the LCD as display with 128 columns and 8 rows (banks). So X ranges from 0 to 127 and Y ranges from 0 to 7. You can refer to the controllers data sheet for more information on this.

When using the _frame buffer_ library, call `renderAll()` after you finish doing all your writing. Alternatively, you can write directly to the display using `renderString()`. When printing multiple lines at a time, using `print()` (rather than `renderString()`) may be quicker.

`setPixel` takes 'x', 'y' and value parameters (0 clears the pixel anything else sets it).

`writeLine` takes x1, y1, x2, y2. As mentioned this only works on horizontal or vertical lines.

`writeRect` takes x, y, width, height, fill (true/false, default is false).

These last 3 methods assume the screen is 128x64.

## Methods

````c
PCD8544_SPI	lcd; // Declares a non-FrameBuffered instance.
PCD8544_SPI_FB	lcd; // Declares a FrameBuffered instance.

// Call a render method after any print/write methods are called.
// For best perofrmance aggragate all writes before calling a render method.
void renderAll();
void renderString(uint8_t x, uint8_t y, uint16_t length);

void setPixel(uint8_t x, uint8_t y, uint8_t value);

// WriteLine currently only supports horizontal and vertical lines.
void writeLine(uint8_t x1, uint8_t y1, uint8_t x2, uint8_t y2);
void writeRect(uint8_t x, uint8_t y, uint8_t width, uint8_t height, bool fill = false);
// Methods above are only available in the FrameBuffer version.

void begin();
void clear();
void gotoXY(uint8_t x, uint8_t y);
virtual size_t write(uint8_t uint8_t);
void writeBitmap(const uint8_t *bitmap, uint8_t x, uint8_t y, uint8_t width, uint8_t height);
````

## Example

````c
#include <SPI.h>
#include <SSD1306_SPI.h>

//#define USE_FRAME_BUFFER

#ifdef USE_FRAME_BUFFER
SSD1306_SPI_FB oled;
#else
SSD1306_SPI oled;
#endif

void setup(void)
{
	oled.begin();
}

void loop()
{
	oled.gotoXY(0, 0);
	oled.print("Hello  ");
#ifdef USE_FRAME_BUFFER
	oled.renderAll();
#endif
	delay(1000);
	oled.gotoXY(0, 0);
	oled.print("SSD1306");
#ifdef USE_FRAME_BUFFER
	oled.renderAll();
#endif
	delay(1000);
}

````
